﻿using Hl7.Fhir.Model;
using Hl7.Fhir.Serialization;
using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Bundle = Hl7.Fhir.Model.Bundle;

namespace VA.PPMS.IWS.Api.Data
{
    internal class FhirMessage
    {
        public Resource Resource { get; protected set; }

        public string FhirMessagePayload { get; protected set; }

        public string Id => Resource != null ? Resource.Id : "No ID";

        public ResourceType ResourceType => Resource?.ResourceType ?? ResourceType.Basic;

        public Bundle AsBundle => Resource as Bundle;

        public DocumentReference AsDocumentReference => Resource as DocumentReference;

        public OperationOutcome AsOperationOutcome => Resource as OperationOutcome;

        public bool IsValid => Resource != null;

        public async Task LoadFhirMessage(HttpRequest request, Encoding encoding)
        {
            Resource = await GetRequestFhirMessage(request, encoding);
        }

        public async Task LoadFhirMessage(string fhirMessage)
        {
            Resource = ParseFhirMessage(fhirMessage);
            await Task.Run(() => { });
        }

        public static async Task<bool> IsMessageValid(HttpRequest request, Encoding encoding)
        {
            var returnObj = new FhirMessage();
            await returnObj.LoadFhirMessage(request, encoding);
            return returnObj.IsValid;
        }

        public string ExtractUrl()
        {
            DocumentReference.ContentComponent content = null;

            switch (ResourceType)
            {
                case ResourceType.Bundle:
                    content = ExtractFirstContent((Bundle)Resource);
                    break;
                case ResourceType.DocumentReference:
                    content = ExtractFirstContent((DocumentReference)Resource);
                    break;
            }

            return ExtractUrl(content);
        }

        private async Task<Resource> GetRequestFhirMessage(HttpRequest request, Encoding encoding)
        {
            Resource returnObj;

            if (encoding == null) encoding = Encoding.UTF8;

            if (request == null) return null;

            using (var reader = new StreamReader(request.Body, encoding))
            {
                FhirMessagePayload = await reader.ReadToEndAsync();
                returnObj = ParseFhirMessage(FhirMessagePayload);
            }

            return returnObj;
        }

        private Resource ParseFhirMessage()
        {
            return ParseFhirMessage(FhirMessagePayload);
        }

        private static Resource ParseFhirMessage(string fhirMessage)
        {
            Resource returnObj = null;

            var fhirObject = Deserialize<FhirObject>(fhirMessage);

            var parser = new FhirJsonParser(new ParserSettings { AcceptUnknownMembers = true, AllowUnrecognizedEnums = true, DisallowXsiAttributesOnRoot = false });

            switch (fhirObject.resourceType)
            {
                case "Bundle":
                    var bundle = parser.Parse<Bundle>(fhirMessage);
                    returnObj = bundle;
                    break;
                case "DocumentReference":
                    var documentReference = parser.Parse<DocumentReference>(fhirMessage);
                    returnObj = documentReference;
                    break;
                case "OperationOutcome":
                    var operationOutcome = parser.Parse<OperationOutcome>(fhirMessage);
                    returnObj = operationOutcome;
                    break;
            }

            return returnObj;
        }

        private static FhirObject Deserialize<FhirObject>(string jsonText)
        {
            byte[] jsonBytes = Encoding.UTF8.GetBytes(jsonText);
            using (var stream = new MemoryStream(jsonBytes))
            {
                return Deserialize<FhirObject>(stream);
            }
        }

        /// <summary>
        /// New code that should pass Fortify scan
        /// </summary>
        /// <typeparam name="FhirObject"></typeparam>
        /// <param name="responseStream"></param>
        /// <returns></returns>
        private static FhirObject Deserialize<FhirObject>(Stream responseStream)
        {
            using (var sr = new StreamReader(responseStream))
            {
                using (var reader = new JsonTextReader(sr))
                {
                    var serializer = new JsonSerializer
                    {
                        MissingMemberHandling = MissingMemberHandling.Ignore,
                        NullValueHandling = NullValueHandling.Ignore
                    };

                    return serializer.Deserialize<FhirObject>(reader);
                }
            }
        }

        private static string ExtractUrl(DocumentReference.ContentComponent content)
        {
            if (!string.IsNullOrEmpty(content?.Attachment?.Url))
            {
                return content.Attachment.Url;
            }

            return string.Empty;
        }

        private DocumentReference.ContentComponent ExtractFirstContent(Bundle bundle)
        {
            if (bundle == null || !bundle.Entry.Any()) return null;

            var docRef = bundle.Entry[0].Resource as DocumentReference;

            return ExtractFirstContent(docRef);
        }

        private static DocumentReference.ContentComponent ExtractFirstContent(DocumentReference element)
        {
            if (element == null || !element.Content.Any()) return null;

            return element.Content.FirstOrDefault();
        }
    }
}